/*****************************************************************************\
*   tty_control - Controlling inputs from serial lines                        *
*   Up to 7 Commands by switches or 4 digital inputs on a serial line         *
*   A work based on: cpanel                                                   *
*                                                                             *
*   Copyright (C) 2000 by Mirko Doelle, DG2FER <dg2fer@qsl.net>               *
*   Modifications and Changes of the origanl Code:(C) 2001-2002               *
*   by Peter Pfrang  <peter.pfrang@gmx.de>                                    *
*   Version 2.00  history see readme                                          *
*                                                                             *
*   This program is free software; you can redistribute it and/or modify      *
*   it under the terms of the GNU General Public License as published by      *
*   the Free Software Foundation; either version 2 of the License, or         *
*   (at your option) any later version.                                       *
*                                                                             *
*   This program is distributed in the hope that it will be useful,           *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of            *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
*   GNU General Public License for more details.                              *
*                                                                             *
*   You should have received a copy of the GNU General Public License         *
*   along with this program; if not, write to the Free Software Foundation,   *
*   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.            *
\*****************************************************************************/

#include "tty_control.h" 


typedef enum
{
  RED = 0, GREEN, YELLOW, GREENYELLOW
} Colour;


/* Signal-Handler: Set Panel-Led to red, write Log message and exit */
void caught_signal(int code)
/* static void caught_signal(int code,int *fd) */
{
  int fd = 0;
  int status = 0;
  ioctl(fd, TIOCMSET, &status); /* Set Panel LED to red */
  close (fd);
	openlog (PROG_NAME, LOG_PID, LOG_LOCAL0); /*Generate Log messages */
	syslog(LOG_INFO,"Process terminated. Caught Signal: %d",code);
	closelog();
  exit(0);
}

int SetPanelLED(Colour LEDs, int fd, int currstat)
{
  if((LEDs & YELLOW) != 0)
    currstat |= TIOCM_RTS;
  else
    currstat &= ~TIOCM_RTS;

  if((LEDs & GREEN) != 0)
    currstat |= TIOCM_DTR;
  else
    currstat &= ~TIOCM_DTR;

  ioctl(fd, TIOCMSET, &currstat);

  return currstat;
}

int BlinkError(int fd, int currstat)
{
  int i;

  for(i=0; i<3; i++)
    {
      usleep(300000);
      SetPanelLED(RED | YELLOW, fd, currstat);
      usleep(300000);
      SetPanelLED(GREEN | YELLOW, fd, currstat);
    }
  return currstat;
}

/* Initialisation of the commands to be executed */
int init_commands(char commands[][COMMAND_MAX_LENGTH+1])
{
  int i;
  char zahl [2];
  FILE* fp;
  char name [COMMAND_MAX_LENGTH];

	for (i=1;i<=7;i++) /* from button 1 to 7 */
	{
	strncpy (name,BUTTONPATH,COMMAND_MAX_LENGTH-1); /* construct string for the command of the button */
	sprintf (zahl,"%d",i); /* print the integer into a string */
	strncat (name,zahl,1); /* append the string (containing the integer) to the BUTTONPATH */

	strncpy (commands[i],name,COMMAND_MAX_LENGTH); /* initialize command string */

	/* script for the button exists */
	if ( (fp=fopen(name,"r")) != NULL)
	  {
	  fclose(fp); /* close filepointer, there is nothing else to be done */
	  }
	else
	  {
	  /* initialisation of the default commands */
	  /* if the script for the button does not exists, then _default is added */
	  /* initialisation of the button1 default command */
	   if (i == 1)
		 strncpy (commands[1],BUTTON1_DEFAULT_COMMAND,COMMAND_MAX_LENGTH);
	   strncat (commands[i],"_default",COMMAND_MAX_LENGTH-strlen(commands[i])); /* default button command */
	  }
	}
	return 0;
}
/* End initialisation of the commands to be executed */

/* Control of the CDROM
   Returns:	0:  on success
		-1: can not open DEVICE
		-2: unable to mount CD
		-3: error while closing tray
		-4: unable to open tray, without any CD in CD-ROM 
		-5: unable to eject CD  */
int cd_control(void)
{
  static int trayopen = 0;
  int cdrom;
  int errorcode=0;

  if((cdrom = open(CDROM_DEV, O_RDONLY | O_NONBLOCK)) < 0)
    {
      /* can not open CD-ROM device */
      return (-1);
    }

  if (ioctl(cdrom, CDROMSTART) != 0)
    {
      /* no CD in CDROM or tray is open */
      if (trayopen == 1)
	{
	  /* tray open => close tray and mount CD */
	  if(ioctl(cdrom, CDROMCLOSETRAY) == 0)
	    {
	      trayopen = 0;
	      sleep(1);
	      if (system_like("CD_ROM_MOUNT") != 0)
		{
		  /* unable to mount CD */
		  errorcode = -2;
		}
	    }
	  else
	    {
	      /* error while closing tray */
		  errorcode = -3;
	    }
	}
      else
	{
	  /* No CD => open tray */
	  if (ioctl(cdrom, CDROMEJECT) == 0)
	    {
	      trayopen = 1;
	    }
	  else
	    {
	      /* unable to open tray */
		  errorcode = -4;
	    }
	}
    }
  else
    {
      /* CD mounted => umount and eject */
      system_like("CD_ROM_UMOUNT");
      if(ioctl(cdrom, CDROMEJECT) == 0)
	{
	  trayopen = 1;
	}
      else
	{
	  /* unable to eject CD */
		  errorcode = -5;
	}
    }
  close(cdrom);
  return(errorcode);
}


/* Panel-control */
void PanelControl(int fd, int currstat, char commands[][COMMAND_MAX_LENGTH+1])
{
  int laststat = currstat;
  int changed = 1;
  int i = 0;
  int success = 0;
  int button_no = 0;

#ifdef DEBUG /* for debugging */
	for (i=1;i<=7;i++) /* from button 1 to 7 */
	{
	printf ("PanelControl Button: %d. Befehl: -%s-\n",i,commands[i]);
	}
#endif

  while(1)
    {
      usleep(75000);                        /* 75ms Delay */
      ioctl(fd, TIOCMGET, &currstat);       /* Read status of the device */
      currstat &= ~(TIOCM_DTR | TIOCM_RTS); /* Cut off LED-status */

      if(currstat != laststat)
	{
	  laststat = currstat ;
	  i = 0;
	  changed = 0;
	}
      else
	{
	  if(i < 2)
	    {
	      i++;
	    }
	  else if(changed == 0)
	    {
	    /* 2 * Delay: Button pressed for 2 times the delay time.*/
	    /* Action will be performed now. */
	    /* DTR und RTS (LED) are ignored, will be reseted  */
	      changed = 1;

	      switch(currstat)
		{
/* Numbering of Buttons according to the ASCII Scheme included in the readme file: S1 = Button 1 */
		case TIOCM_CAR:
		  /* Button 1 pressed */
		  button_no=1;
		  break;
		case TIOCM_CTS:
		  /* Button 2 pressed */
		  button_no=2;
		  break;
		case TIOCM_DSR:
		  /* Button 3 pressed */
		  button_no=3;
		  break;
		case TIOCM_RNG:
		  /* Button 4 pressed */
		  button_no=4;
		  break;
		case (TIOCM_CTS | TIOCM_DSR):
		  /* Button 5 pressed */
		  button_no=5;
		  /*command="nothing 1";*/
		  break;
		case (TIOCM_CTS | TIOCM_CAR):
		  /* Button 6 pressed */
		  button_no=6;
		  break;
		case (TIOCM_DSR | TIOCM_CAR):
		  /* Button 7 pressed */
		  button_no=7;
		  break;
		default:
		  button_no=0;
		  SetPanelLED(GREEN, fd, currstat);
		}

	 #ifdef DEBUG
	   printf ("PanelControl Button: %d pressed\n",button_no);
	 #endif
	      if (button_no != 0)
	      {
	      /* Set panel led yellow and green */
	      SetPanelLED(GREEN | YELLOW, fd, currstat);

	        openlog (PROG_NAME, LOG_PID, LOG_LOCAL0);
	        syslog(LOG_INFO,"Button %d: executing command: %s",button_no,commands[button_no]);
	        closelog();

/* Check if the defaultcommand for cd-rom should be executed (function and not a system call)*/
	      if (strcmp(commands[button_no], BUTTON1_DEFAULT_COMMAND) == 0)
		{
		 /* Call the subroutine for the control of the cd-rom */
		success = cd_control();
		}
		else
		{
		/* system call for the command */
		/* former version: success=system(commands[button_no]); */
		success=system_like(commands[button_no]);
		}

		if (success != 0)
		{ /* Execution of the command failed => blinking led, error log */
		  BlinkError(fd, currstat);

	          openlog (PROG_NAME, LOG_PID, LOG_LOCAL0);
	          syslog(LOG_INFO,"%s returned error code: %d",commands[button_no],success);
	          closelog();
		}
		else
		{ /* execution has been successful */
	          openlog (PROG_NAME, LOG_PID, LOG_LOCAL0);
	          syslog(LOG_INFO,"Execution of %s was successful",commands[button_no]);
	          closelog();
		}

		  button_no=0;
		  /* Set panel green only*/
		  SetPanelLED(GREEN, fd, currstat);
	      }


	     }

	}
    }
}

int main(int argc, char** argv)
{
  int i;
  int status = 0;
  int fd=0;
  int currstat = 0;
  char commands[8][COMMAND_MAX_LENGTH+1];
  /* char commands[8][COMMAND_MAX_LENGTH]='\0'; */
/* Initialization of the commands-array: last charachter is end of string */
 for (i=0; i<8;i++)
    commands[i][COMMAND_MAX_LENGTH]='\0';


  /* Check Parameters: serial device has to be the first parameter */
  if(argc != 2)
    {
	openlog (PROG_NAME, LOG_PID, LOG_LOCAL0);
	syslog(LOG_INFO,"Error: %s: too few arguments. Syntax: %s <device>",argv[0],argv[0]);
	closelog();
      return 1;
    }

  /* open Device */
  if((fd = open(argv[1], O_RDWR | O_NONBLOCK | O_EXCL)) < 0)
    {
	openlog (PROG_NAME, LOG_PID, LOG_LOCAL0);
	syslog(LOG_INFO,"Error: opening device \"%s\" failed",argv[1]);
	closelog();
      return 2;
    } 

  /* Handle Signals: Set Panel-LED red, write log message and exit */
  signal (SIGINT, caught_signal);
  signal (SIGHUP, caught_signal);
  signal (SIGTERM, caught_signal);
  signal (SIGABRT, caught_signal);
  signal (SIGILL, caught_signal);
  signal (SIGSEGV, caught_signal);
  signal (SIGTRAP, caught_signal);
  signal (SIGUSR1, caught_signal);
  signal (SIGUSR2, caught_signal);
/*
  signal (SIGINT, caught_signal(SIGINT,&fd));
  signal (SIGHUP, caught_signal(SIGHUP,&fd));
  signal (SIGTERM, caught_signal(SIGTERM,&fd));
  signal (SIGUSR2, init_commands(commands)); */

  
/* Start process as Daemon */
status = to_daemon();

	openlog (PROG_NAME, LOG_PID, LOG_LOCAL0);
	syslog(LOG_INFO,"Started as Daemon: Version "VERSION);
	closelog();

if ( status != 0)
{
	openlog (PROG_NAME, LOG_PID, LOG_LOCAL0);
	syslog(LOG_INFO,"Error: to_daemon not succesfull");
	closelog();
   return 1;
}
/* End of to Daemon */

  /* Initialisation of the command array */
status = init_commands (commands);
if ( status != 0)
{
	openlog (PROG_NAME, LOG_PID, LOG_LOCAL0);
	syslog(LOG_INFO,"Error:Initialisation of the commands failed");
	closelog();
    return 1;
}

  /* Panel initialisation: green */
  currstat = TIOCM_DTR;
  ioctl(fd, TIOCMSET, &currstat);


  /* call the subroutine for the control of the panel, normally there is no return in this function */
  PanelControl(fd, currstat, commands);

  return 0;
}
