/*
 * portato.c - implementation of configurable parallel port LED status
 *             meter for the Linux OS
 *
 * see the README file which accompanies the Portato distribution
 * for details.
 *
 * author: Peter Heist (heistp@rpi.edu)
 * version: 1.2
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <errno.h>

#ifdef PROFILE
#include <time.h>
#include <sys/time.h>
#endif

#ifndef TESTCOMPILE
#include "port.h"
#endif

#ifdef TESTCOMPILE
#define DEVFILE "dev"
#define STATFILE "stat"
#else
#define DEVFILE "/proc/net/dev"
#define STATFILE "/proc/stat"
#endif

/* the types of activities portato can monitor for */
#define ACT_CPUIDL    0     /* cpu idle */
#define ACT_CPUNCE    1     /* nice cpu */
#define ACT_CPUSYS    2     /* system cpu */
#define ACT_CPUUSR    3     /* user cpu */
#define ACT_DISK1     4     /* disk activity 1 */
#define ACT_DISK2     5     /* disk activity 2 */
#define ACT_DISK3     6     /* disk activity 3 */
#define ACT_DISK4     7     /* disk activity 4 */
#define ACT_INT1      8     /* interrupt 1 */
#define ACT_INT2      9     /* interrupt 2 */
#define ACT_INT3      10    /* interrupt 3 */
#define ACT_INT4      11    /* interrupt 4 */
#define ACT_INT5      12    /* interrupt 5 */
#define ACT_INT6      13    /* interrupt 6 */
#define ACT_INT7      14    /* interrupt 7 */
#define ACT_INT8      15    /* interrupt 8 */
#define ACT_INT9      16    /* interrupt 9 */
#define ACT_INT10     17    /* interrupt 10 */
#define ACT_INT11     18    /* interrupt 11 */
#define ACT_INT12     19    /* interrupt 12 */
#define ACT_INT13     20    /* interrupt 13 */
#define ACT_INT14     21    /* interrupt 14 */
#define ACT_INT15     22    /* interrupt 15 */
#define ACT_NETCOL    23    /* net packets colliding */
#define ACT_NETRCV    24    /* net packets being received */
#define ACT_NETSND    25    /* net packets being sent */
#define ACT_SWPIN     26    /* pages swapping in */
#define ACT_SWPOUT    27    /* pages swapping out */

typedef struct portato_activity_tag {
   int activity_type;
   int line; /* line in file where this data is */
   unsigned short *value_ptr; /* exactly where in the file the value is */
   unsigned short val; /* current value of this resource */
   unsigned char bitmask; /* value to be or'd out to parallel port */
   unsigned char **buffer; /* where the buffer pointer is for this data */
} portato_activity;

/* activity list structure */
typedef struct activity_list_node_tag {
   portato_activity pa;
   struct activity_list_node_tag *next;
   struct activity_list_node_tag *prev;
} activity_list_node;

typedef struct activity_list_tag {
   activity_list_node *head;
   activity_list_node *tail;
   activity_list_node *current;
} activity_list;

/* global variables (hey, why modularize?  this is free!) */
char config_file[255] = CONFIG_FILE;
char inbuf[150];
static int line = 0;
unsigned int port_address = 0; /* address of port to write to */
unsigned long poll_delay = 0; /* time between updates (microseconds) */
activity_list global_al;
unsigned char *devbuf = NULL;
unsigned char *statbuf = NULL;
int dev_size = 0;
int stat_size = 0;
int need_dev = 0;
int need_stat = 0;
portato_activity *act_array = NULL;
int num_activities = 0;
#ifdef PROFILE
clock_t cputime = 0;
struct timeval start_tv;
#endif

/* the following two structures dictate where particular elements are
   located in each of the proc files */
struct portato_activity_type
{
   int field; /* which field this activity is in */
   int type; /* 1 for stat, 2 for dev */
};

struct portato_activity_type portato_activity_types [] =
{
   {	4,  1	}, /* ACT_CPUIDL */
   {	2,  1	}, /* ACT_CPUNCE */
   {	3,  1	}, /* ACT_CPUSYS */
   {	1,  1	}, /* ACT_CPUUSR */
   {	1,  1	}, /* ACT_DISK1 */
   {	2,  1	}, /* ACT_DISK2 */
   {	3,  1	}, /* ACT_DISK3 */
   {	4,  1	}, /* ACT_DISK4 */
   {	3,  1	}, /* ACT_INT1 */
   {	4,  1	}, /* ACT_INT2 */
   {	5,  1	}, /* ACT_INT3 */
   {	6,  1	}, /* ACT_INT4 */
   {	7,  1	}, /* ACT_INT5 */
   {	8,  1	}, /* ACT_INT6 */
   {	9,  1	}, /* ACT_INT7 */
   {	10, 1	}, /* ACT_INT8 */
   {	11, 1	}, /* ACT_INT9 */
   {	12, 1	}, /* ACT_INT10 */
   {	13, 1	}, /* ACT_INT11 */
   {	14, 1	}, /* ACT_INT12 */
   {	15, 1	}, /* ACT_INT13 */
   {	16, 1	}, /* ACT_INT14 */
   {	17, 1	}, /* ACT_INT15 */
   {	10, 2	}, /* ACT_NETCOL */
   {	1,  2	}, /* ACT_NETRCV */
   {	6,  2	}, /* ACT_NETSND */
   {	1,  1	}, /* ACT_SWPIN */
   {	2,  1	}  /* ACT_SWPOUT */
};

void initialize_activity_list(activity_list *al)
{
   al->head = al->tail = al->current = NULL;
}


void destroy_activity_list(activity_list *al)
{
   activity_list_node *cap;
   activity_list_node *newcap;

   if (al == NULL) return;

   cap = al->head;
   while (cap != NULL)
   {
      newcap = cap->next;
      free(cap);
      cap = newcap;
   }
}


void add_activity_list_node(activity_list *al, int activity_type,
   int line, unsigned char bm)
{
   activity_list_node *ap;
   activity_list_node *cap;
   activity_list_node *oldcap;

   ap = (activity_list_node *) malloc(sizeof(activity_list_node));
   ap->pa.activity_type = activity_type;
   ap->pa.line = line;
   ap->pa.val = 0x00;
   ap->pa.bitmask = bm;
   switch (portato_activity_types[activity_type].type)
   {
      case 1:
	ap->pa.buffer = &statbuf;
	need_stat = 1;
	break;
      case 2:
	ap->pa.buffer = &devbuf;
	need_dev = 1;
	break;
   }

   ap->next = ap->prev = NULL;
   if (al->head == NULL) /* if list is empty add unconditionally */
      { al->head = al->tail = ap; return; }
   /* otherwise add prioritized */
   oldcap = NULL;
   cap = al->head;
   while(cap != NULL)
   {
      if (ap->pa.line < cap->pa.line ||
	((ap->pa.line == cap->pa.line) &&
	 (portato_activity_types[ap->pa.activity_type].field <
	  portato_activity_types[cap->pa.activity_type].field)) )
      {
	if (cap == al->head)
	   { ap->next = al->head; al->head = ap; return; }
	else
	   { oldcap->next = ap; ap->next = cap; return; }
      }

      oldcap = cap;
      cap = cap->next;
   }
   al->tail->next = ap;
   al->tail = ap;
}


void reset_activity_list_peek(activity_list *al)
{
   al->current = al->head;
}


portato_activity *activity_list_peek(activity_list *al)
{
   portato_activity *return_ptr;

   return_ptr = &al->current->pa;

   if (al->current != NULL)
      al->current = al->current->next;

   return return_ptr;
}


void create_activity_array(portato_activity **act_array)
{
   portato_activity *act;

   num_activities = 0;

   reset_activity_list_peek(&global_al);
   while((act = activity_list_peek(&global_al)) != NULL)
   {
      num_activities++;
      if (((*act_array) = realloc((*act_array), num_activities *
	sizeof(portato_activity))) == NULL)
	{ perror("failed realloc"); exit(1); }
      bcopy(act, (&(*act_array)[num_activities - 1]),
	sizeof(portato_activity));
   }
}


void portato_shutdown(int signum)
{
#ifdef PROFILE
   struct timeval end_tv;
   double total_cputime = ((double) clock() - (double) cputime) /
      (double) CLOCKS_PER_SEC;
   long total_seconds;

   gettimeofday(&end_tv, NULL);
   total_seconds = end_tv.tv_sec - start_tv.tv_sec;
   printf("Portato was active for %ld seconds\n", total_seconds);
   printf("CPU time used = %.2f sec (roughly %.2f%%)\n", total_cputime,
      100 * total_cputime / (double) total_seconds);
#endif
   destroy_activity_list(&global_al);
   if (devbuf != NULL) free(devbuf);
   if (statbuf != NULL) free(statbuf);
#ifndef TESTCOMPILE
   port_out(port_address, 0);
#endif
   exit(0);
}


void file_error()
{
   fprintf(stderr, "portato: some sort of error on line %d in the config file, "
      "text as follows:\n", line);
   fprintf(stderr, "portato: %s", inbuf);
   destroy_activity_list(&global_al);
   exit(1);
}

int get_next_line(FILE *fp)
{
   int retval;
   char arg1[30];
   char arg2[30];

   while(fgets(inbuf, 256, fp) != NULL)
   {
      line++;
      retval = sscanf(inbuf, "%s %s", arg1, arg2);
      if (strncmp(arg1, ";", 1) && retval == 2) return 1;
   }
   return 0;
}


void read_portato_config_file(void)
{
   int drive;
   int irq;
   int pin;
   int line;
   char arg1[80];
   char arg2[80];
   char arg3[80];
   char proc_arg1[30];
   char proc_arg2[30];
   char inbuf2[150];
   char device[30];
   unsigned char bm = 0x00;
   FILE *cf_fp;
   FILE *net_fp;
   FILE *stat_fp;
   int found_device;

   if ((cf_fp = fopen(config_file, "r")) == NULL)
      { fprintf(stderr, "config file (%s) not found\n", config_file); exit(1); }

   if ((net_fp = fopen(DEVFILE, "r")) == NULL)
      { fprintf(stderr, "%s not found\n", DEVFILE); exit(1); }

   if ((stat_fp = fopen(STATFILE, "r")) == NULL)
      { fprintf(stderr, "%s not found\n", STATFILE); exit(1); }

   /* initialize activity list */
   initialize_activity_list(&global_al);

   /* port section */
   if (get_next_line(cf_fp) == 0) file_error();
   if (sscanf(inbuf, "port %s %s", arg1, arg2) != 2) file_error();
   if (!strcmp(arg1, "lpt"))
   {
      if (atoi(arg2) == 1) port_address = 0x378;
      else if (atoi(arg2) == 2) port_address = 0x278;
      else file_error();
   }
   else if (!strcmp(arg1, "address"))
   {
      sscanf(arg2, "%x", &port_address);
   }
   else file_error();

#ifndef TESTCOMPILE
   /* set access permissions on the port */
   if ((ioperm(port_address, 1, 1)) == -1)
   {
      fprintf(stderr, "unable to set permissions on port 0x%x\n",
	port_address);
      perror("portato");
      exit(1);
   }
#endif

   /* update rate section */
   if (get_next_line(cf_fp) == 0) file_error();
   if (sscanf(inbuf, "update %s %s", arg1, arg2) != 2) file_error();
   if (atoi(arg1) < 0) file_error();
   if (!strcmp(arg2, "sec")) poll_delay = atoi(arg1) * 1000000;
   else if (!strcmp(arg2, "msec")) poll_delay = atoi(arg1) * 1000;
   else if (!strcmp(arg2, "usec")) poll_delay = atoi(arg1);
   else file_error();

   /* pin section */
   while(get_next_line(cf_fp))
   {
      if (sscanf(inbuf, "pin %s %s %s", arg1, arg2, arg3) != 3) file_error();
      pin = atoi(arg1);
      switch (pin)
      {
	case 1: bm = 0x01; break;
	case 2: bm = 0x02; break;
	case 3: bm = 0x04; break;
	case 4: bm = 0x08; break;
	case 5: bm = 0x10; break;
	case 6: bm = 0x20; break;
	case 7: bm = 0x40; break;
	case 8: bm = 0x80; break;
	default: fprintf(stderr, "unexpected pin\n"); file_error();
	   break;
      }
      if ( !strcmp(arg2, "cpu") || !strcmp(arg2, "disk") ||
	!strcmp(arg2, "intr") || !strcmp(arg2, "swap") )
      {
	line = 1;
	rewind(stat_fp);
	while(fgets(inbuf2, 255, stat_fp) != NULL)
	{
	   sscanf(inbuf2, "%s %s", proc_arg1, proc_arg2);
	   if (strcmp(arg2, proc_arg1)) { line++; continue; }
	   if (!strcmp(arg2, "cpu"))
	   {
	      if (!strcmp(arg3, "user"))
		add_activity_list_node(&global_al, ACT_CPUUSR, line, bm);
	      else if (!strcmp(arg3, "system"))
		add_activity_list_node(&global_al, ACT_CPUSYS, line, bm);
	      else if (!strcmp(arg3, "nice"))
		add_activity_list_node(&global_al, ACT_CPUNCE, line, bm);
	      else if (!strcmp(arg3, "activity"))
	      {
		add_activity_list_node(&global_al, ACT_CPUUSR, line, bm);
		add_activity_list_node(&global_al, ACT_CPUSYS, line, bm);
		add_activity_list_node(&global_al, ACT_CPUNCE, line, bm);
	      }
	      else if (!strcmp(arg3, "idle"))
		add_activity_list_node(&global_al, ACT_CPUIDL, line, bm);
	      else file_error();
	   }
	   else if (!strcmp(arg2, "disk"))
	   {
	      drive = atoi(arg3);
	      switch(drive)
	      {
		case 0:
		   add_activity_list_node(&global_al, ACT_DISK1, line, bm);
		   add_activity_list_node(&global_al, ACT_DISK2, line, bm);
		   add_activity_list_node(&global_al, ACT_DISK3, line, bm);
		   add_activity_list_node(&global_al, ACT_DISK4, line, bm);
		   break;
		case 1:
		   add_activity_list_node(&global_al, ACT_DISK1, line, bm);
		   break;
		case 2:
		   add_activity_list_node(&global_al, ACT_DISK2, line, bm);
		   break;
		case 3:
		   add_activity_list_node(&global_al, ACT_DISK3, line, bm);
		   break;
		case 4:
		   add_activity_list_node(&global_al, ACT_DISK4, line, bm);
		   break;
		default:
		   file_error();
		   break;
	      }
	   }
	   else if (!strcmp(arg2, "intr"))
	   {
	      irq = atoi(arg3);
	      switch(irq)
	      {
		case 1:
		   add_activity_list_node(&global_al, ACT_INT1, line, bm);
		   break;
		case 2:
		   add_activity_list_node(&global_al, ACT_INT2, line, bm);
		   break;
		case 3:
		   add_activity_list_node(&global_al, ACT_INT3, line, bm);
		   break;
		case 4:
		   add_activity_list_node(&global_al, ACT_INT4, line, bm);
		   break;
		case 5:
		   add_activity_list_node(&global_al, ACT_INT5, line, bm);
		   break;
		case 6:
		   add_activity_list_node(&global_al, ACT_INT6, line, bm);
		   break;
		case 7:
		   add_activity_list_node(&global_al, ACT_INT7, line, bm);
		   break;
		case 8:
		   add_activity_list_node(&global_al, ACT_INT8, line, bm);
		   break;
		case 9:
		   add_activity_list_node(&global_al, ACT_INT9, line, bm);
		   break;
		case 10:
		   add_activity_list_node(&global_al, ACT_INT10, line, bm);
		   break;
		case 11:
		   add_activity_list_node(&global_al, ACT_INT11, line, bm);
		   break;
		case 12:
		   add_activity_list_node(&global_al, ACT_INT12, line, bm);
		   break;
		case 13:
		   add_activity_list_node(&global_al, ACT_INT13, line, bm);
		   break;
		case 14:
		   add_activity_list_node(&global_al, ACT_INT14, line, bm);
		   break;
		case 15:
		   add_activity_list_node(&global_al, ACT_INT15, line, bm);
		   break;
		default:
		   file_error();
		   break;
	      }
	   }
	   else if (!strcmp(arg2, "swap"))
	   {
	      if (!strcmp(arg3, "in"))
		add_activity_list_node(&global_al, ACT_SWPIN, line, bm);
	      else if (!strcmp(arg3, "out"))
		add_activity_list_node(&global_al, ACT_SWPOUT, line, bm);
	      else if (!strcmp(arg3, "activity"))
	      {
		add_activity_list_node(&global_al, ACT_SWPIN, line, bm);
		add_activity_list_node(&global_al, ACT_SWPOUT, line, bm);
	      }
	      else file_error();
	   }
	   else file_error();
	}
      }
      else if ( !strcmp(arg2, "net_activity") ||
	!strcmp(arg2, "net_collisions") ||
	!strcmp(arg2, "net_receive") ||
	!strcmp(arg2, "net_transmit") )
      {
	line = 1;
	found_device = 0;
	rewind(net_fp);
	while(fgets(inbuf2, 255, net_fp) != NULL)
	{
	   sscanf(inbuf2, "%s:", device);
	   if (!strncmp(device, arg3, strlen(arg3)))
	   {
	      if (!strcmp(arg2, "net_activity"))
	      {
		add_activity_list_node(&global_al, ACT_NETRCV, line, bm);
		add_activity_list_node(&global_al, ACT_NETSND, line, bm);
	      }
	      else if (!strcmp(arg2, "net_collisions"))
		add_activity_list_node(&global_al, ACT_NETCOL, line, bm);
	      else if (!strcmp(arg2, "net_receive"))
		add_activity_list_node(&global_al, ACT_NETRCV, line, bm);
	      else if (!strcmp(arg2, "net_transmit"))
		add_activity_list_node(&global_al, ACT_NETSND, line, bm);
	      else file_error();
	      found_device = 1;
	   }
	   line++;
	}
	if (!found_device) file_error();
      }
      else file_error();
   }
   fclose(net_fp);
   fclose(stat_fp);
   fclose(cf_fp);
}


void recalculate_value_ptrs(int type)
{
   unsigned char *bufptr;
   unsigned char *final_bufptr;
   register int i;
   register int fields;
   register int newlines;
   int target_newline;
   int target_field;

   if (type == 1) { bufptr = statbuf; final_bufptr = bufptr + stat_size; }
   else { bufptr = devbuf; final_bufptr = bufptr + dev_size; }

   newlines = 1;
   fields = 0;
   for(i = 0; i < num_activities; i++)
   {
      if(portato_activity_types[act_array[i].activity_type].type == type)
      {
	/* set target line and field */
	target_newline = act_array[i].line;
	target_field =
	   portato_activity_types[act_array[i].activity_type].field;

	/* seek to line */
	if (newlines != target_newline)
	{
	   while(bufptr < final_bufptr)
	   {
	      /* count number of newlines */
	      if (*bufptr == '\n')
		{ newlines++; if (newlines == target_newline)
		   { bufptr++; fields = 0; break; } }
	      bufptr++;
	   }
	}

	/* seek to proper part of field and assign value_ptr */
	while(bufptr < final_bufptr)
	{
	   /* if field zero then seek to it */
	   if (fields == 0)
	   {
	      while(bufptr < final_bufptr)
		{ if (*bufptr != ' ') break; bufptr++; }
	      while(bufptr < final_bufptr)
		{ if (*bufptr == ' ' || *bufptr == '\n') break; 
		   if (*bufptr == ':') { bufptr++; break; }
		   bufptr++; }
	   }

	   if (fields == target_field)
	      { act_array[i].value_ptr =
		   (unsigned short *) (bufptr - sizeof(unsigned short));
		break; }

	   while(bufptr < final_bufptr)
	      { if (*bufptr != ' ') break; bufptr++; }
	   while(bufptr < final_bufptr)
	      { if (*bufptr == ' ' || *bufptr == '\n') break;
		if (*bufptr == ':') { bufptr++; break; }
		bufptr++; }

	   fields++;
	}
      }
   }
}


int main_loop(void)
{
   int dev_fd = 0;
   int stat_fd = 0;
   register unsigned char port_value = 0x00;
   register int i;
   int old_file_size;
#ifdef DEBUG
   unsigned char *bufptr;
#endif
   register unsigned short oldval;

   /* set up buffers for reading the procfiles */
   if (need_dev) devbuf = malloc(4096 * sizeof(unsigned char));
   if (need_stat) statbuf = malloc(1024 * sizeof(unsigned char));

#ifdef PROFILE
   cputime = clock();
   gettimeofday(&start_tv, NULL);
#endif

   while(1)
   {
      if (need_dev)
      {
	dev_fd = open(DEVFILE, O_RDONLY);
	old_file_size = dev_size;
	dev_size = read(dev_fd, devbuf, 4096);
	if (old_file_size != dev_size)
	   recalculate_value_ptrs(2);
	close(dev_fd);
      }

      if (need_stat)
      {
	stat_fd = open(STATFILE, O_RDONLY);
	old_file_size = stat_size;
	stat_size = read(stat_fd, statbuf, 1024);
	if (old_file_size != stat_size)
	   recalculate_value_ptrs(1);
	close(stat_fd);
      }
#ifdef DEBUG
      for (i = 0; i < num_activities; i++)
      {
	if (portato_activity_types[act_array[i].activity_type].type == 1)
	   bufptr = statbuf;
	else bufptr = devbuf;
	printf("type = %d, line = %d, field = %d, offset = %d\n",
	   act_array[i].activity_type, act_array[i].line,
	   portato_activity_types[act_array[i].activity_type].field,
	   (int) (unsigned char *) act_array[i].value_ptr - (int) bufptr);
      }
#endif

      port_value = 0x00;

      for(i = 0; i < num_activities; i++)
      {
	oldval = act_array[i].val;

	/* update new value and check it with old one */
	act_array[i].val = *(act_array[i].value_ptr);

	if (oldval != act_array[i].val)
	   port_value |= act_array[i].bitmask;
      }

#ifndef TESTCOMPILE
      port_out(port_address, port_value);
#endif
      usleep(poll_delay);
   }
}


int main(int argc, char **argv)
{
   if (argc > 1)
      strcpy(config_file, argv[1]);

   /* invoke our signal handlers */
   signal(SIGTERM, portato_shutdown);
   signal(SIGINT, portato_shutdown);

   /* read the config file and assign the resource list */
   read_portato_config_file();

   /* take linked list and create an array of activities for fast indexing */
   create_activity_array(&act_array);

   /* don't need the linked list any longer */
   destroy_activity_list(&global_al);

   /* poll system 'till the cows come home (or we receive a SIGINT) */
   main_loop();
   exit(0);
}
