/* ledblink - let your LEDs blink
 * Copyright (C) 2002 Jrg Mensmann <joerg.mensmann@gmx.de>
 *
 * Based very much on mailblink 0.7
 * Copyright (C) 2001 Martin Renold
 * Copyright (C) 1999 Donald J Bindner
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 */

/* I would like to acknowledge Jouni.Lohikoski@iki.fi whose utility
 * tleds influenced some of the look and feel for this program and
 * whose source code served as my example for process-handling.
 * (by Donald)
 */

/* I just added POP3/IMAP support by stealing pop.* and socklib.*
 * from asmail-0.56. It was quiet easy, thanks to MoGul
 * (by Martin)
 */

/* Removed all threading and mailchecking support - now its more a
 * blink-server (by Jrg)
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>

#include <signal.h>
#include <errno.h>

#include <unistd.h>
#include <dirent.h>

#define VERSION "0.85"
#define MYNAME "ledblink"
#define PIDFILE "/var/run/ledblink.pid"
#define LOCKFILE "/var/run/.ledblink-lock"

#define WARN 0
#define MESSAGE 1

static const char *light[11] = { "num", "caps", "scroll", "parport0", "parport1", 
				 "parport2", "parport3", "parport4", "parport5",
				 "parport6", "parport7" };

/* These control blinking */
extern int lednum;
extern int blinks;
extern int led_on;
extern void *led_loop( void );
extern void *led_exit( void );

/* file descriptor used to lock pid-file */
static int fd;
/* pids for each light */
static pid_t pid[11] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

/* saved euid */
static uid_t saved_euid;

/* some user options */
static int opt_b=0, opt_d=0, opt_s=0, opt_k=0, opt_D=0, opt_0=0, opt_1=0, opt_t=0;

/* default verbosity */
int verbosity = 1;

void help_exit( void ) {
    printf( "%s version %s\t\tJrg Mensmann <joerg.mensmann@gmx.de>\n\
Usage: %s [options]\n\
    -b\t\tBlink more\n\
    -d\t\tDecrease blinking rate\n\
    -s\t\tStop blinking\n\
    -1\t\tPut led into on state\n\
    -0\t\tPut led into off state\n\
    -t\t\tToggle led state\n\
    -k\t\tKill old processes.\n\
    -l <led>\tLed to blink (num, caps or scroll, parport0 - parport7)\n\
    -q\t\tDecrease verbosity.  Repeat for even less.\n\
    -v\t\tIncrease verbosity.  Repeat for even more.\n\
    -D\t\tDebug.  Don't fork into the background.\n", MYNAME, VERSION, MYNAME );
    exit( 1 );
}

void message( int level, const char *format, ... ) {
    va_list ap;

    if( verbosity < level ) return;

    va_start( ap, format );
    vfprintf( stdout, format, ap );
    fflush( stdout );
}

int pid_file_exists( void ) {
    struct stat status;

    message( 3, "Checking for existing PID file.\n" );
    return !( stat( PIDFILE, &status ));
}

int lock_pid_file( void ) {
    message( 3, "Locking PID file.\n" );
    return fd = open( LOCKFILE, O_WRONLY|O_CREAT|O_EXCL|O_SYNC, 0666);
}

void unlock_pid_file( void ) {
    message( 3, "Unlocking PID file.\n" );
    assert( fd != -1 );
    if( close( fd )) assert(0);
    if( unlink( LOCKFILE )) assert( 0 );
}

void read_pid_file( void ) {
    char str[255];
    int i;
    FILE *f;

    message( 3, "Reading PID file.\n" );

    if( (f = fopen( PIDFILE, "r" )) != NULL) {
	for( i = 0; i < 11; i++ ) {
	    fgets( str, sizeof(str), f );
	    pid[i] = strtol( str, NULL, 10 );
	}
	fclose( f );
    }
}

void write_pid_file( void ) {
    int i;
    FILE *f;

    message( 3, "Writing PID file.\n" );

    if( (f = fopen( PIDFILE, "w" ) ) != 0) {
	for( i = 0; i < 11; i++ ) {
	    fprintf( f, "%d %s\n", pid[i], light[i] );
	}
	fclose( f );
    }
}

void kill_old_process( int lednum ) {
/*
 * Will kill the (old) process who's PID is read from a
 * valid PID file.
 */

    message( 2, "Killing old process: %d.\n", pid[lednum] );

    /* Check first for process to kill */
    if( !pid[lednum] ) {
	message( MESSAGE, "No process to kill.\n" );
	return;
    }

    /* lower privilege */
    seteuid( getuid());

    if( kill( pid[lednum], SIGTERM )) {
	switch( errno ) {
	    case ESRCH:		/* process was already gone */
		pid[lednum] = 0;
		break;
	    case EPERM:
		message( WARN, "Permission denied killing PID %d.\n", pid[lednum] );
		break;
	    default:
		assert( 0 );
	}
    } else {
	message( MESSAGE, "Process %ld killed.\n", pid[lednum] );
	pid[lednum] = 0;
    }

    /* recover privilege */
    seteuid( saved_euid );
}

void send_sig( int lednum, int sig ) {
/*
 * Will send signal to the process who's PID is read from a
 * valid PID file.
 */

    /* Check first for process to sig */
    if( !pid[lednum] ) {
	message( MESSAGE, "No process to send signal to. Start server without options first.\n" );
	return;
    }

    /* lower privilege */
    seteuid( getuid());

    if( kill( pid[lednum], sig )) {
	switch( errno ) {
	    case ESRCH:		/* process was already gone */
		pid[lednum] = 0;
		break;
	    case EPERM:
		message( WARN, "Permission denied sending sig to PID %d.\n", pid[lednum] );
		break;
	    default:
		assert( 0 );
	}
    } 

    /* recover privilege */
    seteuid( saved_euid );
}


int old_process_alive( int lednum ) {
    char procFileName[256];
    struct stat status;

    message( 3, "Checking process status for PID: %d.\n", pid[lednum] );

    sprintf( procFileName, "/proc/%ld/environ", (long)pid[lednum] );
    return( !stat(procFileName, &status ));
}

void remove_pid() {
    message( 3, "Removing PID from PID file: %d.\n", pid[lednum] );

    if( lock_pid_file() == -1 ) exit( 1 );

    read_pid_file();
    pid[lednum] = 0;
    write_pid_file();
    unlock_pid_file();
}

void my_exit( void ) {
    blinks = 0;

    message( 3, "my_exit() called\n" );

    led_exit();

    /* recover privilege */
    seteuid( saved_euid );

    remove_pid();
}

/* Signal handlers */

/* SIGHUP - stop blinking */
void hup_handler(int signum) {
  message( 2, "Received HUP - stopped blinking.\n" );
  blinks = 0;
}

/* SIGUSR1 - blink more often */
void usr1_handler(int signum) {
  message( 2, "Received USR1 - blinking more.\n" );
  blinks++;
}

/* SIGUSR2 - blink less often */
void usr2_handler(int signum) {
  message( 2, "Received USR2 - blinking less.\n" );
  if (blinks > 0) blinks--;
}

/* SIGALRM - led on */
void alrm_handler(int signum) {
  message( 2, "Received ALRM - led on.\n" );
  led_on = 1;
}

/* SIGTSTP - led off */
void tstp_handler(int signum) {
  message( 2, "Received TSTP - led off.\n" );
  led_on = 0;
}

/* SIGCHLD - led toggle */
void chld_handler(int signum) {
  message( 2, "Received CHLD - led toggle.\n" );
  led_on = !led_on;
}

void my_handler( int sig ) {
    exit( 1 );
}

int main( int argc, char *argv[] ) {
    int c, i;
    pid_t childpid;
    sigset_t sigset;
    struct sigaction sa;

    /* So we can drop privilege temporarily */
    saved_euid = geteuid();
    seteuid( saved_euid );
    
    lednum = 2; /* default to blink scroll lock */

    /* block signals from delivery until we are ready to handle them */
    if( sigemptyset( &sigset )) assert(0);
    if( sigaddset( &sigset, SIGTERM )) assert(0);
    if( sigaddset( &sigset, SIGINT )) assert(0);
    if( sigaddset( &sigset, SIGHUP )) assert(0);
    if( sigaddset( &sigset, SIGQUIT )) assert(0);
    if( sigaddset( &sigset, SIGTTOU )) assert(0);
    if( sigprocmask( SIG_BLOCK, &sigset, NULL )) assert(0);

    /* Get all of the command line switches */
    while( EOF != ( c = getopt( argc, argv, "Dbdskhl:qv01t" ))) {
	switch( c ) {
	    case 'D':
		opt_D = 1;
		break;
	    case 'h':
		help_exit();
		break;
	    case 'b':
  	        opt_b = 1;
		break;
	    case 'd':
  	        opt_d = 1;
		break;
	    case 's':
  	        opt_s = 1;
		break;
	    case 'k':
		opt_k = 1;
		break;
	    case 'l':
		lednum = -1;
		for( i = 0; i < 11; i++ ) {
		    if( !strcmp( optarg, light[i] )) lednum = i;
		}
		if( lednum == -1 ) help_exit();
		break;
	    case 'q':
		verbosity--;	/* lower verbosity */
		break;
	    case 'v':
		verbosity++;	/* increase verbosity */
		break;
   	    case '0':
		opt_0 = 1;
		break;
   	    case '1':
		opt_1 = 1;
		break;
   	    case 't':
	        opt_t = 1;
		break;
	    default:
		help_exit();
	}
    }

    /* Might as well get this out of the way */
    if( geteuid()) {
	message( WARN, "This program requires root privilege to run.\n" );
	exit( 1 );
    }

    /* Lock pid file and read it -- DON'T EXIT UNTIL THIS LOCK IS REMOVED */
    if( lock_pid_file() == -1 ) {
	message( WARN, "Could not create lockfile: %s\n", LOCKFILE );
	exit( 1 );
    }
    read_pid_file();

    /* kill an existing process */
    if( opt_k ) {
	kill_old_process( lednum );
	write_pid_file();
	unlock_pid_file();
	exit( 0 );
    }

    /* blink more */
    if( opt_b ) {
        message( 2, "Blink more.\n" );
	send_sig( lednum, SIGUSR1 );
	unlock_pid_file();
	exit( 0 );
    }
    /* blink less */
    if( opt_d ) {
        message( 2, "Blink less.\n" );
	send_sig( lednum, SIGUSR2 );
	unlock_pid_file();
	exit( 0 );
    }
    /* stop blinking */
    if( opt_s ) {
        message( 2, "Stop blinking.\n" );
	send_sig( lednum, SIGHUP );
	unlock_pid_file();
	exit( 0 );
    }
    /* led on */
    if( opt_1 ) {
        message( 2, "LED on.\n" );
	send_sig( lednum, SIGALRM );
	unlock_pid_file();
	exit( 0 );
    }    
    /* led off */
    if( opt_0 ) {
        message( 2, "LED off.\n" );
	send_sig( lednum, SIGTSTP );
	unlock_pid_file();
	exit( 0 );
    }    
    /* led toggle */
    if( opt_t ) {
        message( 2, "LED toggle.\n" );
	send_sig( lednum, SIGCHLD );
	unlock_pid_file();
	exit( 0 );
    }    

    /* someone may be already using this led */
    if( pid[lednum] ) {
	if( old_process_alive( lednum )) {
	    message( WARN, "Led %s already used by PID: %d\n", 
		    light[lednum], pid[lednum] );
	    unlock_pid_file();
	    exit( 1 );
	} else {
	    /* process was dead */
	    message( 2, "Old pid was already dead: %d\n", pid[lednum] );
	    pid[lednum] = 0;
	    write_pid_file();
	}
    }

    if( opt_D ) {	/* Run in foreground to debug */
	message( MESSAGE, "Debug mode--running in foreground.\n" );
	childpid = getpid();
    } else {	/* fork into the background */
	if( (childpid = fork()) == -1 ) {
	    message( WARN, "%s: fork() failed", MYNAME );

	    unlock_pid_file();
	    exit( 1 );
	}
    }

    if( childpid ) {	/* parent code */
	/* update the pid file */
	pid[lednum] = childpid;
	write_pid_file();
	unlock_pid_file();

	message( MESSAGE, "%s running with PID %ld.\n", MYNAME, childpid );
	if( !opt_D ) exit( 0 );		/* only exit if not debugging */
    }

    /* detatch ourself from a controlling terminal and mounted filesystems */
    if( !opt_D ) {	/* if not debugging */
	setsid();
	chdir( "/" );
	umask( 0 );
    }

  
    /* Set up our exit protocol */
    if( atexit(my_exit)) assert(0);

    /* set signal handler and deliver all blocked signals */
    {
	struct sigaction act;

	memset( &act, 0, sizeof( struct sigaction ));
	act.sa_handler = my_handler;
	act.sa_flags |= SA_RESTART;
	sigfillset( &act.sa_mask );
	if( sigaction( SIGTERM, &act, NULL )) assert(0);
	if( sigaction( SIGINT, &act, NULL )) assert(0);
	if( sigaction( SIGHUP, &act, NULL )) assert(0);
	if( sigaction( SIGQUIT, &act, NULL )) assert(0);
	if( sigaction( SIGTTOU, &act, NULL )) assert(0);
	if( sigprocmask( SIG_UNBLOCK, &sigset, NULL )) assert(0);
    }

    /* Set up sighandlers */

    memset(&sa, 0, sizeof(sa)); 
    sa.sa_handler = hup_handler;
    sigaction(SIGHUP, &sa, NULL);

    memset(&sa, 0, sizeof(sa)); 
    sa.sa_handler = usr1_handler;
    sigaction(SIGUSR1, &sa, NULL);

    memset(&sa, 0, sizeof(sa)); 
    sa.sa_handler = usr2_handler;
    sigaction(SIGUSR2, &sa, NULL);

    memset(&sa, 0, sizeof(sa)); 
    sa.sa_handler = alrm_handler;
    sigaction(SIGALRM, &sa, NULL);

    memset(&sa, 0, sizeof(sa)); 
    sa.sa_handler = tstp_handler;
    sigaction(SIGTSTP, &sa, NULL);

    memset(&sa, 0, sizeof(sa)); 
    sa.sa_handler = chld_handler;
    sigaction(SIGCHLD, &sa, NULL);

    blinks = 0;

    /* continue indefinitely */
    while( 1 ) {
      led_loop();
    }

    exit( 0 );
}
